﻿<# Copyright © 2023. Cloud Software Group, Inc. All Rights Reserved. #>
# In older versions of PowerShell (before Windows 10), Get-Service did not return StartType. To support these older operating systems, we have to switch to WMI when we detect that class doesn't have this custom property. 
# We cannot use WMI call for all operating systems though, because it doesn't see some of the services (most notably kernel driver services like 'CSC')
[Boolean]$m_IsServicesLegacyMode = $(Get-Service | Select-Object -First 1).StartType -isnot [Object];

# Function to test is service with specified name exists. 
Function Test-CTXOEServicesServiceExist ([String]$Name) {
    Return $(Get-Service -Name $Name -ErrorAction SilentlyContinue) -is [System.ServiceProcess.ServiceController]
}

# Function to test if specified service has desider startup type configured. 
Function Test-CTXOEServicesStartupType ([String]$Name, [String]$StartType) {
    [Hashtable]$Return = @{}

    $Return.Result = $False

    # Test if service exists. 
    If ($(Test-CTXOEServicesServiceExist -Name $Name) -eq $False) {
        $Return.Details = "Service $($Params.Name) was not found"
        $Return.NotFound = $true;
        $Return.Result = $StartType -eq "Disabled";
        Return $Return
    } Else {
        If ($m_IsServicesLegacyMode) {
            # If legacy service mode is detected, Get-Service does not support start mode and we have to switch to WMI mode instead. However, WMI is not using 'Automatic', but shortened version 'Auto'. If 'Auto' is detected in output, we need to change it right variant ("Automatic")
            [String]$m_LegacyStartMode = (Get-WmiObject Win32_Service -filter "Name='$Name'").StartMode;
            If ($m_LegacyStartMode -eq "Auto") {
                [String]$m_CurrentStartMode = "Automatic";
            } Else {
                [String]$m_CurrentStartMode = $m_LegacyStartMode;
            }
        } Else {
            [String]$m_CurrentStartMode = Get-Service -Name $Name | Select-Object -ExpandProperty StartType;
        }

        $Return.Result = $m_CurrentStartMode -eq $StartType

        If ($m_CurrentStartMode -eq $StartType) {
            $Return.Details = "Startup type is $StartType"
        } Else {
            $Return.Details = "Desired startup type $($StartType), current type $($m_CurrentStartMode)"
        }
    }

    Return $Return
}

Function Invoke-CTXOEServicesExecuteInternal ([Xml.XmlElement]$Params, [Boolean]$RollbackSupported = $False) {
    [Hashtable]$m_Result = Test-CTXOEServicesStartupType -Name $Params.Name -StartType $Params.Value

    # If service is already configured, no action need to be taken.
    If ($m_Result.Result -eq $true) {
        $Global:CTXOE_Result = $m_Result.Result
        $Global:CTXOE_Details = $m_Result.Details
        Return
    }

    # If service was not found, return
    If ($m_Result.NotFound) {
        # If service should be disabled AND does not exists, that's OK. But if other value was configured than DISABLED, it's not OK. 
        $Global:CTXOE_Result = $Params.Value -eq "Disabled"
        $Global:CTXOE_Details = "Service $($Params.Name) was not found"
        Return
    } Else {
        # Save the current startup mode. 
        If ($m_IsServicesLegacyMode) {
            # If legacy service mode is detected, Get-Service does not support start mode and we have to switch to WMI mode instead. However, WMI is not using 'Automatic', but shortened version 'Auto'. If 'Auto' is detected in output, we need to change it right variant ("Automatic")
            [String]$m_PreviousLegacyStartMode = (Get-WmiObject Win32_Service -filter "Name='$($Params.Name)'").StartMode;
            If ($m_PreviousLegacyStartMode -eq "Auto") {
                [String]$m_PreviousStartMode = "Automatic";
            } Else {
                [String]$m_PreviousStartMode = $m_PreviousLegacyStartMode;
            }
        } Else {
            [String]$m_PreviousStartMode = Get-Service -Name $($Params.Name) | Select-Object -ExpandProperty StartType;
        }

        # Set service to required type, re-run the test for startup type
        Try {
            # There is a bug in Set-Service - it cannot properly set start type SYSTEM (used for kernel driver services). Instead of using cmdlet, we will change it directly in registry
            If ($Params.Value -eq "System") {
                New-ItemProperty -Path Registry::HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Services\$($Params.Name) -Name Start -PropertyType DWORD -Value 1 -Force | Out-Null
            } Else {
                If ($Params.Value -eq "Auto") { # WMI is using 'Auto', Set-Service is using "Automatic"
                    Set-Service -Name $Params.Name -StartupType Automatic
                } Else {
                    Set-Service -Name $Params.Name -StartupType $Params.Value
                }
            }

            # Try to stop service. This is not critical action, only nice to have, so we don't need to check if it was successful or not (or even if service exists)
            If ($Params.Value -eq "Disabled") {Stop-Service $Params.Name -ErrorAction SilentlyContinue;}
        } Catch {
            $Global:CTXOE_Result = $False; 
            $Global:CTXOE_Details = "Failed to change service state with following error: $($_.Exception.Message)"; 
            Return
        }

        # Same the last known startup type for rollback instructions
        $Global:CTXOE_SystemChanged = $true;
        If ($RollbackSupported) {
            [Xml.XmlDocument]$m_RollbackElement = CTXOE\ConvertTo-CTXOERollbackElement -Element $Params
            $m_RollbackElement.rollbackparams.value = $m_PreviousStartMode.ToString();
            $Global:CTXOE_ChangeRollbackParams = $m_RollbackElement
        }

        [Hashtable]$m_Result = Test-CTXOEServicesStartupType -Name $Params.Name -StartType $Params.Value
        $Global:CTXOE_Result = $m_Result.Result
        $Global:CTXOE_Details = $m_Result.Details
        # There is a bug in Set-Service - it cannot properly set start type SYSTEM (used for kernel driver services). We change it in registry instead, but this is reflected after reboot, so we cannot use normal status reporting.
        If ($Params.Value -eq "System") {
            $Global:CTXOE_Result = $True;
            $Global:CTXOE_Details = "Startup type is System"
        }
    }

    Return
}

Function Invoke-CTXOEServicesAnalyze ([Xml.XmlElement]$Params) {
    [Hashtable]$m_Result = Test-CTXOEServicesStartupType -Name $Params.Name -StartType $Params.Value

    $Global:CTXOE_Result = $m_Result.Result
    $Global:CTXOE_Details = $m_Result.Details

    Return

}

Function Invoke-CTXOEServicesExecute ([Xml.XmlElement]$Params) {
    Invoke-CTXOEServicesExecuteInternal -Params $Params -RollbackSupported $True
}

Function Invoke-CTXOEServicesRollback ([Xml.XmlElement]$Params) {
    Invoke-CTXOEServicesExecuteInternal -Params $Params -RollbackSupported $False
}
# SIG # Begin signature block
# MIIoqQYJKoZIhvcNAQcCoIIomjCCKJYCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCA1pEQuf1E9wTTs
# 53oOk4+z9/EeCqWzpGa4/plmW0oB1qCCDcAwggawMIIEmKADAgECAhAIrUCyYNKc
# TJ9ezam9k67ZMA0GCSqGSIb3DQEBDAUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQK
# EwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNV
# BAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yMTA0MjkwMDAwMDBaFw0z
# NjA0MjgyMzU5NTlaMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAw
# ggIKAoICAQDVtC9C0CiteLdd1TlZG7GIQvUzjOs9gZdwxbvEhSYwn6SOaNhc9es0
# JAfhS0/TeEP0F9ce2vnS1WcaUk8OoVf8iJnBkcyBAz5NcCRks43iCH00fUyAVxJr
# Q5qZ8sU7H/Lvy0daE6ZMswEgJfMQ04uy+wjwiuCdCcBlp/qYgEk1hz1RGeiQIXhF
# LqGfLOEYwhrMxe6TSXBCMo/7xuoc82VokaJNTIIRSFJo3hC9FFdd6BgTZcV/sk+F
# LEikVoQ11vkunKoAFdE3/hoGlMJ8yOobMubKwvSnowMOdKWvObarYBLj6Na59zHh
# 3K3kGKDYwSNHR7OhD26jq22YBoMbt2pnLdK9RBqSEIGPsDsJ18ebMlrC/2pgVItJ
# wZPt4bRc4G/rJvmM1bL5OBDm6s6R9b7T+2+TYTRcvJNFKIM2KmYoX7BzzosmJQay
# g9Rc9hUZTO1i4F4z8ujo7AqnsAMrkbI2eb73rQgedaZlzLvjSFDzd5Ea/ttQokbI
# YViY9XwCFjyDKK05huzUtw1T0PhH5nUwjewwk3YUpltLXXRhTT8SkXbev1jLchAp
# QfDVxW0mdmgRQRNYmtwmKwH0iU1Z23jPgUo+QEdfyYFQc4UQIyFZYIpkVMHMIRro
# OBl8ZhzNeDhFMJlP/2NPTLuqDQhTQXxYPUez+rbsjDIJAsxsPAxWEQIDAQABo4IB
# WTCCAVUwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUaDfg67Y7+F8Rhvv+
# YXsIiGX0TkIwHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0P
# AQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHcGCCsGAQUFBwEBBGswaTAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAC
# hjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9v
# dEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5j
# b20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAcBgNVHSAEFTATMAcGBWeBDAED
# MAgGBmeBDAEEATANBgkqhkiG9w0BAQwFAAOCAgEAOiNEPY0Idu6PvDqZ01bgAhql
# +Eg08yy25nRm95RysQDKr2wwJxMSnpBEn0v9nqN8JtU3vDpdSG2V1T9J9Ce7FoFF
# UP2cvbaF4HZ+N3HLIvdaqpDP9ZNq4+sg0dVQeYiaiorBtr2hSBh+3NiAGhEZGM1h
# mYFW9snjdufE5BtfQ/g+lP92OT2e1JnPSt0o618moZVYSNUa/tcnP/2Q0XaG3Ryw
# YFzzDaju4ImhvTnhOE7abrs2nfvlIVNaw8rpavGiPttDuDPITzgUkpn13c5Ubdld
# AhQfQDN8A+KVssIhdXNSy0bYxDQcoqVLjc1vdjcshT8azibpGL6QB7BDf5WIIIJw
# 8MzK7/0pNVwfiThV9zeKiwmhywvpMRr/LhlcOXHhvpynCgbWJme3kuZOX956rEnP
# LqR0kq3bPKSchh/jwVYbKyP/j7XqiHtwa+aguv06P0WmxOgWkVKLQcBIhEuWTatE
# QOON8BUozu3xGFYHKi8QxAwIZDwzj64ojDzLj4gLDb879M4ee47vtevLt/B3E+bn
# KD+sEq6lLyJsQfmCXBVmzGwOysWGw/YmMwwHS6DTBwJqakAwSEs0qFEgu60bhQji
# WQ1tygVQK+pKHJ6l/aCnHwZ05/LWUpD9r4VIIflXO7ScA+2GRfS0YW6/aOImYIbq
# yK+p/pQd52MbOoZWeE4wggcIMIIE8KADAgECAhAExKVRXTiJBY6ZBfjtmlFwMA0G
# CSqGSIb3DQEBCwUAMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwHhcNMjQwMzA4MDAwMDAwWhcNMjUwMzA3
# MjM1OTU5WjCBjzELMAkGA1UEBhMCVVMxEDAOBgNVBAgTB0Zsb3JpZGExGDAWBgNV
# BAcTD0ZvcnQgTGF1ZGVyZGFsZTEdMBsGA1UEChMUQ2l0cml4IFN5c3RlbXMsIElu
# Yy4xFjAUBgNVBAsTDUNpdHJpeCBYZW5BcHAxHTAbBgNVBAMTFENpdHJpeCBTeXN0
# ZW1zLCBJbmMuMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEA+gJ84IK2
# t0aBgBuXB3yfqoC+riwIEohXTQHVqNvJ7v3xiiIYD8PvzYLhaQq2dG4euY1AOhXL
# 0Uc2rNLcQZKEB4PslYrrorg+x5iRJl2k9XltDEpDbqJJqkUELY6d7GEIpp4ChdGf
# NJuGU9GCAgiml88tzZMOjtAqcLIXSRSOugQgjUOR+EvfeLlMTLs/ADjyOi898+0m
# TtdE6Sy/a2bn/fgsJaU8O2Kn78YiHX1N6lklKKBzO5tYfRD2VZhOn2kx6PUI+n+R
# ckwE22o0hBhckBeBGNW2DUrABDLwYuNOEWZEGx1KC2UDO4p+eDmxANjC/+q9t53d
# Z4V1Y/qL9CRnin0JNowVcLT3zGqeebgIag2Pqsh6m3lvASiy7DpmDtyaTCwv/aLd
# 1x5TZvUPiKcfvMOSCRxTP+a7L4DdjuqcQxWENk1kNEZjZa/ZS0BDWalklp6Ji9ki
# JpUKxB5C6GD/Lns+hbdp/2XHnEsqGbeQBCiW7ftXO3ugffq+/Bz1CelPAgMBAAGj
# ggIDMIIB/zAfBgNVHSMEGDAWgBRoN+Drtjv4XxGG+/5hewiIZfROQjAdBgNVHQ4E
# FgQUh00+M0JEuzdBGq4PW6l3UIoBQsIwPgYDVR0gBDcwNTAzBgZngQwBBAEwKTAn
# BggrBgEFBQcCARYbaHR0cDovL3d3dy5kaWdpY2VydC5jb20vQ1BTMA4GA1UdDwEB
# /wQEAwIHgDATBgNVHSUEDDAKBggrBgEFBQcDAzCBtQYDVR0fBIGtMIGqMFOgUaBP
# hk1odHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRDb2Rl
# U2lnbmluZ1JTQTQwOTZTSEEzODQyMDIxQ0ExLmNybDBToFGgT4ZNaHR0cDovL2Ny
# bDQuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNpZ25pbmdSU0E0
# MDk2U0hBMzg0MjAyMUNBMS5jcmwwgZQGCCsGAQUFBwEBBIGHMIGEMCQGCCsGAQUF
# BzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5jb20wXAYIKwYBBQUHMAKGUGh0dHA6
# Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNENvZGVTaWdu
# aW5nUlNBNDA5NlNIQTM4NDIwMjFDQTEuY3J0MAkGA1UdEwQCMAAwDQYJKoZIhvcN
# AQELBQADggIBAI1e2H/UCCob4N14i0h2Z1Yi7dInt7RICuwc9yiMcDeGvlvtgcJT
# zp/CyQwpDh58/WGxR0HX3TuLjLh0Foxzm6aSgTLnyvI65jugv24L7yu1Bl5n7WR1
# x4tqtvVI1DOCTWNN9BqokeBTuT7NVwDPyoLWxz8f3GqnERJ/xe4DJOv9RXre6sy0
# NF9a+d4oqvEJ5r03WOpI6jO3YG6hyxCF7611ZhVdvBuw0ZfVgXPH1n/l+kGb4Mdc
# s0kJ6ny/auQ3Cm0lpCxrjh+LxwrxYtk1s/iZMd9T+C5vjUzH1LWhqlPur0ur05yU
# cG61A8pPwKLE02FdW9b6miAPknpUI6o6EgdU1KRCCjtYoR1iBSagB75k0RZBcgor
# B6mQKcFF5EwRhw5rSJdn1r7rTjRgVTzgRjgHL7QBY27yshp9rTwqHzO9VUmfTxg1
# 4UT5dcZ3xGc8U4j+gMSjNXmJJEpRaNtuh1QDXoQCwC0fM/9r4BxIlS50nuWrhnBL
# zSbrdxbmDK+iW5UU2oCKs3zHbrPmqQgWd4CDi/g/kUEruRIL+HMS1eAdZlyZ8DCw
# H8Qwv6+T6we/iDDbHtxLy7TjGsYChqEgaVOT7gj++iIdoHq3+mb1bxtDgovELXFN
# EwOxXT5mt7qZlc4ApoUTdruNzS9aMofoIB2SIVo/P6FCryDE5YxVmv6zMYIaPzCC
# GjsCAQEwfTBpMQswCQYDVQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4x
# QTA/BgNVBAMTOERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5nIFJTQTQw
# OTYgU0hBMzg0IDIwMjEgQ0ExAhAExKVRXTiJBY6ZBfjtmlFwMA0GCWCGSAFlAwQC
# AQUAoIHQMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsx
# DjAMBgorBgEEAYI3AgEVMC8GCSqGSIb3DQEJBDEiBCCoD0/MzDNH+5ix2JVHG50t
# T3uT4eF41Hn9dglIR8ACqDBkBgorBgEEAYI3AgEMMVYwVKA4gDYAQwBpAHQAcgBp
# AHgAIABTAHUAcABwAG8AcgB0AGEAYgBpAGwAaQB0AHkAIABUAG8AbwBsAHOhGIAW
# aHR0cDovL3d3dy5jaXRyaXguY29tIDANBgkqhkiG9w0BAQEFAASCAYCeNFTgLVYZ
# Es63Bn/CoK77sBs/TXgDDqfwa7GcX4x2zVsS6gJsSx3IwK/xYQN3ifU+Ct9sEB2m
# 8TlgXzfQqxF73NAoIDkATryapqyH2x5KZbPa0npbhn3/euCV/Z6Oxsu3mS24iFI4
# jLUjpGdhS2miaQ3UGtDfTrDXusmWBwrcFrjs+sYAcnUMfWfYvyhBzGKWpN4iZYux
# +N7HWLRirOpmuM20d4VzK+OpWyqTEfpNgv6+TKlC4Av3Z4TJ2cCx3COadQVKHTgn
# 4bosZOjAg/chOMnwIkBSziYkBaYCM8h6QZLNb4cy1BcnkxFzirHdtr3ulYaYfK7k
# joIJu73moG/2VQySPP+1Ib4YTr+kOIaSYHhmnLqIsfCfB0TQHsE/QhQH0PEDlDk6
# gPuAVkiCUkjH8xkFgZRKGyLEwkyKxwc05IdLFyygAKPqFrWpIl7+pAzvjpDDewco
# OU51+luRYUeY/3Fe8Q+he2QoMkIq4zDIfA3IwqpRwm2nk4Ld/jHFxSmhghdAMIIX
# PAYKKwYBBAGCNwMDATGCFywwghcoBgkqhkiG9w0BBwKgghcZMIIXFQIBAzEPMA0G
# CWCGSAFlAwQCAQUAMHgGCyqGSIb3DQEJEAEEoGkEZzBlAgEBBglghkgBhv1sBwEw
# MTANBglghkgBZQMEAgEFAAQgY9gdFdA2z2nlENR1TvyzToLA87cGMNP1ag4/M8aP
# 48ICEQC1umWfkS8aiqwgtEoNz63bGA8yMDI0MDMyMTEwMDAyNlqgghMJMIIGwjCC
# BKqgAwIBAgIQBUSv85SdCDmmv9s/X+VhFjANBgkqhkiG9w0BAQsFADBjMQswCQYD
# VQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lD
# ZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMB4X
# DTIzMDcxNDAwMDAwMFoXDTM0MTAxMzIzNTk1OVowSDELMAkGA1UEBhMCVVMxFzAV
# BgNVBAoTDkRpZ2lDZXJ0LCBJbmMuMSAwHgYDVQQDExdEaWdpQ2VydCBUaW1lc3Rh
# bXAgMjAyMzCCAiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAKNTRYcdg45b
# rD5UsyPgz5/X5dLnXaEOCdwvSKOXejsqnGfcYhVYwamTEafNqrJq3RApih5iY2nT
# WJw1cb86l+uUUI8cIOrHmjsvlmbjaedp/lvD1isgHMGXlLSlUIHyz8sHpjBoyoNC
# 2vx/CSSUpIIa2mq62DvKXd4ZGIX7ReoNYWyd/nFexAaaPPDFLnkPG2ZS48jWPl/a
# Q9OE9dDH9kgtXkV1lnX+3RChG4PBuOZSlbVH13gpOWvgeFmX40QrStWVzu8IF+qC
# ZE3/I+PKhu60pCFkcOvV5aDaY7Mu6QXuqvYk9R28mxyyt1/f8O52fTGZZUdVnUok
# L6wrl76f5P17cz4y7lI0+9S769SgLDSb495uZBkHNwGRDxy1Uc2qTGaDiGhiu7xB
# G3gZbeTZD+BYQfvYsSzhUa+0rRUGFOpiCBPTaR58ZE2dD9/O0V6MqqtQFcmzyrzX
# xDtoRKOlO0L9c33u3Qr/eTQQfqZcClhMAD6FaXXHg2TWdc2PEnZWpST618RrIbro
# HzSYLzrqawGw9/sqhux7UjipmAmhcbJsca8+uG+W1eEQE/5hRwqM/vC2x9XH3mwk
# 8L9CgsqgcT2ckpMEtGlwJw1Pt7U20clfCKRwo+wK8REuZODLIivK8SgTIUlRfgZm
# 0zu++uuRONhRB8qUt+JQofM604qDy0B7AgMBAAGjggGLMIIBhzAOBgNVHQ8BAf8E
# BAMCB4AwDAYDVR0TAQH/BAIwADAWBgNVHSUBAf8EDDAKBggrBgEFBQcDCDAgBgNV
# HSAEGTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwHwYDVR0jBBgwFoAUuhbZbU2F
# L3MpdpovdYxqII+eyG8wHQYDVR0OBBYEFKW27xPn783QZKHVVqllMaPe1eNJMFoG
# A1UdHwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2Vy
# dFRydXN0ZWRHNFJTQTQwOTZTSEEyNTZUaW1lU3RhbXBpbmdDQS5jcmwwgZAGCCsG
# AQUFBwEBBIGDMIGAMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5kaWdpY2VydC5j
# b20wWAYIKwYBBQUHMAKGTGh0dHA6Ly9jYWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdp
# Q2VydFRydXN0ZWRHNFJTQTQwOTZTSEEyNTZUaW1lU3RhbXBpbmdDQS5jcnQwDQYJ
# KoZIhvcNAQELBQADggIBAIEa1t6gqbWYF7xwjU+KPGic2CX/yyzkzepdIpLsjCIC
# qbjPgKjZ5+PF7SaCinEvGN1Ott5s1+FgnCvt7T1IjrhrunxdvcJhN2hJd6PrkKoS
# 1yeF844ektrCQDifXcigLiV4JZ0qBXqEKZi2V3mP2yZWK7Dzp703DNiYdk9WuVLC
# tp04qYHnbUFcjGnRuSvExnvPnPp44pMadqJpddNQ5EQSviANnqlE0PjlSXcIWiHF
# tM+YlRpUurm8wWkZus8W8oM3NG6wQSbd3lqXTzON1I13fXVFoaVYJmoDRd7ZULVQ
# jK9WvUzF4UbFKNOt50MAcN7MmJ4ZiQPq1JE3701S88lgIcRWR+3aEUuMMsOI5lji
# tts++V+wQtaP4xeR0arAVeOGv6wnLEHQmjNKqDbUuXKWfpd5OEhfysLcPTLfddY2
# Z1qJ+Panx+VPNTwAvb6cKmx5AdzaROY63jg7B145WPR8czFVoIARyxQMfq68/qTr
# eWWqaNYiyjvrmoI1VygWy2nyMpqy0tg6uLFGhmu6F/3Ed2wVbK6rr3M66ElGt9V/
# zLY4wNjsHPW2obhDLN9OTH0eaHDAdwrUAuBcYLso/zjlUlrWrBciI0707NMX+1Br
# /wd3H3GXREHJuEbTbDJ8WC9nR2XlG3O2mflrLAZG70Ee8PBf4NvZrZCARK+AEEGK
# MIIGrjCCBJagAwIBAgIQBzY3tyRUfNhHrP0oZipeWzANBgkqhkiG9w0BAQsFADBi
# MQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3
# d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3Qg
# RzQwHhcNMjIwMzIzMDAwMDAwWhcNMzcwMzIyMjM1OTU5WjBjMQswCQYDVQQGEwJV
# UzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRy
# dXN0ZWQgRzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBMIICIjANBgkq
# hkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAxoY1BkmzwT1ySVFVxyUDxPKRN6mXUaHW
# 0oPRnkyibaCwzIP5WvYRoUQVQl+kiPNo+n3znIkLf50fng8zH1ATCyZzlm34V6gC
# ff1DtITaEfFzsbPuK4CEiiIY3+vaPcQXf6sZKz5C3GeO6lE98NZW1OcoLevTsbV1
# 5x8GZY2UKdPZ7Gnf2ZCHRgB720RBidx8ald68Dd5n12sy+iEZLRS8nZH92GDGd1f
# tFQLIWhuNyG7QKxfst5Kfc71ORJn7w6lY2zkpsUdzTYNXNXmG6jBZHRAp8ByxbpO
# H7G1WE15/tePc5OsLDnipUjW8LAxE6lXKZYnLvWHpo9OdhVVJnCYJn+gGkcgQ+ND
# Y4B7dW4nJZCYOjgRs/b2nuY7W+yB3iIU2YIqx5K/oN7jPqJz+ucfWmyU8lKVEStY
# dEAoq3NDzt9KoRxrOMUp88qqlnNCaJ+2RrOdOqPVA+C/8KI8ykLcGEh/FDTP0kyr
# 75s9/g64ZCr6dSgkQe1CvwWcZklSUPRR8zZJTYsg0ixXNXkrqPNFYLwjjVj33GHe
# k/45wPmyMKVM1+mYSlg+0wOI/rOP015LdhJRk8mMDDtbiiKowSYI+RQQEgN9XyO7
# ZONj4KbhPvbCdLI/Hgl27KtdRnXiYKNYCQEoAA6EVO7O6V3IXjASvUaetdN2udIO
# a5kM0jO0zbECAwEAAaOCAV0wggFZMBIGA1UdEwEB/wQIMAYBAf8CAQAwHQYDVR0O
# BBYEFLoW2W1NhS9zKXaaL3WMaiCPnshvMB8GA1UdIwQYMBaAFOzX44LScV1kTN8u
# Zz/nupiuHA9PMA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAKBggrBgEFBQcDCDB3
# BggrBgEFBQcBAQRrMGkwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0
# LmNvbTBBBggrBgEFBQcwAoY1aHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0Rp
# Z2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcnQwQwYDVR0fBDwwOjA4oDagNIYyaHR0cDov
# L2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcmwwIAYD
# VR0gBBkwFzAIBgZngQwBBAIwCwYJYIZIAYb9bAcBMA0GCSqGSIb3DQEBCwUAA4IC
# AQB9WY7Ak7ZvmKlEIgF+ZtbYIULhsBguEE0TzzBTzr8Y+8dQXeJLKftwig2qKWn8
# acHPHQfpPmDI2AvlXFvXbYf6hCAlNDFnzbYSlm/EUExiHQwIgqgWvalWzxVzjQEi
# Jc6VaT9Hd/tydBTX/6tPiix6q4XNQ1/tYLaqT5Fmniye4Iqs5f2MvGQmh2ySvZ18
# 0HAKfO+ovHVPulr3qRCyXen/KFSJ8NWKcXZl2szwcqMj+sAngkSumScbqyQeJsG3
# 3irr9p6xeZmBo1aGqwpFyd/EjaDnmPv7pp1yr8THwcFqcdnGE4AJxLafzYeHJLtP
# o0m5d2aR8XKc6UsCUqc3fpNTrDsdCEkPlM05et3/JWOZJyw9P2un8WbDQc1PtkCb
# ISFA0LcTJM3cHXg65J6t5TRxktcma+Q4c6umAU+9Pzt4rUyt+8SVe+0KXzM5h0F4
# ejjpnOHdI/0dKNPH+ejxmF/7K9h+8kaddSweJywm228Vex4Ziza4k9Tm8heZWcpw
# 8De/mADfIBZPJ/tgZxahZrrdVcA6KYawmKAr7ZVBtzrVFZgxtGIJDwq9gdkT/r+k
# 0fNX2bwE+oLeMt8EifAAzV3C+dAjfwAL5HYCJtnwZXZCpimHCUcr5n8apIUP/JiW
# 9lVUKx+A+sDyDivl1vupL0QVSucTDh3bNzgaoSv27dZ8/DCCBY0wggR1oAMCAQIC
# EA6bGI750C3n79tQ4ghAGFowDQYJKoZIhvcNAQEMBQAwZTELMAkGA1UEBhMCVVMx
# FTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNv
# bTEkMCIGA1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4XDTIyMDgw
# MTAwMDAwMFoXDTMxMTEwOTIzNTk1OVowYjELMAkGA1UEBhMCVVMxFTATBgNVBAoT
# DERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEhMB8GA1UE
# AxMYRGlnaUNlcnQgVHJ1c3RlZCBSb290IEc0MIICIjANBgkqhkiG9w0BAQEFAAOC
# Ag8AMIICCgKCAgEAv+aQc2jeu+RdSjwwIjBpM+zCpyUuySE98orYWcLhKac9WKt2
# ms2uexuEDcQwH/MbpDgW61bGl20dq7J58soR0uRf1gU8Ug9SH8aeFaV+vp+pVxZZ
# VXKvaJNwwrK6dZlqczKU0RBEEC7fgvMHhOZ0O21x4i0MG+4g1ckgHWMpLc7sXk7I
# k/ghYZs06wXGXuxbGrzryc/NrDRAX7F6Zu53yEioZldXn1RYjgwrt0+nMNlW7sp7
# XeOtyU9e5TXnMcvak17cjo+A2raRmECQecN4x7axxLVqGDgDEI3Y1DekLgV9iPWC
# PhCRcKtVgkEy19sEcypukQF8IUzUvK4bA3VdeGbZOjFEmjNAvwjXWkmkwuapoGfd
# pCe8oU85tRFYF/ckXEaPZPfBaYh2mHY9WV1CdoeJl2l6SPDgohIbZpp0yt5LHucO
# Y67m1O+SkjqePdwA5EUlibaaRBkrfsCUtNJhbesz2cXfSwQAzH0clcOP9yGyshG3
# u3/y1YxwLEFgqrFjGESVGnZifvaAsPvoZKYz0YkH4b235kOkGLimdwHhD5QMIR2y
# VCkliWzlDlJRR3S+Jqy2QXXeeqxfjT/JvNNBERJb5RBQ6zHFynIWIgnffEx1P2Ps
# IV/EIFFrb7GrhotPwtZFX50g/KEexcCPorF+CiaZ9eRpL5gdLfXZqbId5RsCAwEA
# AaOCATowggE2MA8GA1UdEwEB/wQFMAMBAf8wHQYDVR0OBBYEFOzX44LScV1kTN8u
# Zz/nupiuHA9PMB8GA1UdIwQYMBaAFEXroq/0ksuCMS1Ri6enIZ3zbcgPMA4GA1Ud
# DwEB/wQEAwIBhjB5BggrBgEFBQcBAQRtMGswJAYIKwYBBQUHMAGGGGh0dHA6Ly9v
# Y3NwLmRpZ2ljZXJ0LmNvbTBDBggrBgEFBQcwAoY3aHR0cDovL2NhY2VydHMuZGln
# aWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNydDBFBgNVHR8EPjA8
# MDqgOKA2hjRodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVk
# SURSb290Q0EuY3JsMBEGA1UdIAQKMAgwBgYEVR0gADANBgkqhkiG9w0BAQwFAAOC
# AQEAcKC/Q1xV5zhfoKN0Gz22Ftf3v1cHvZqsoYcs7IVeqRq7IviHGmlUIu2kiHdt
# vRoU9BNKei8ttzjv9P+Aufih9/Jy3iS8UgPITtAq3votVs/59PesMHqai7Je1M/R
# Q0SbQyHrlnKhSLSZy51PpwYDE3cnRNTnf+hZqPC/Lwum6fI0POz3A8eHqNJMQBk1
# RmppVLC4oVaO7KTVPeix3P0c2PR3WlxUjG/voVA9/HYJaISfb8rbII01YBwCA8sg
# sKxYoA5AY8WYIsGyWfVVa88nq2x2zm8jLfR+cWojayL/ErhULSd+2DrZ8LaHlv1b
# 0VysGMNNn3O3AamfV6peKOK5lDGCA3YwggNyAgEBMHcwYzELMAkGA1UEBhMCVVMx
# FzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMuMTswOQYDVQQDEzJEaWdpQ2VydCBUcnVz
# dGVkIEc0IFJTQTQwOTYgU0hBMjU2IFRpbWVTdGFtcGluZyBDQQIQBUSv85SdCDmm
# v9s/X+VhFjANBglghkgBZQMEAgEFAKCB0TAaBgkqhkiG9w0BCQMxDQYLKoZIhvcN
# AQkQAQQwHAYJKoZIhvcNAQkFMQ8XDTI0MDMyMTEwMDAyNlowKwYLKoZIhvcNAQkQ
# AgwxHDAaMBgwFgQUZvArMsLCyQ+CXc6qisnGTxmcz0AwLwYJKoZIhvcNAQkEMSIE
# IMA3wkvROO/phIygfzFSl7mO8Mi0dJFa1+pmwpLDRFE+MDcGCyqGSIb3DQEJEAIv
# MSgwJjAkMCIEINL25G3tdCLM0dRAV2hBNm+CitpVmq4zFq9NGprUDHgoMA0GCSqG
# SIb3DQEBAQUABIICAIEibL4iKkn0Kuanacljzv7KL6aR3wd5iVB/CqGH2mWveu1l
# zZnY5gNeECXQRL+m9BIdBrGoIRRyVgnjo/mcg2H5mvL6MHjKsnMKbuxOuzi8lefY
# tWx9UOKORVlk6/ON+moTP0RQqCSmnsD1zgsK50TOQKKg8vcBaMaK6IhbDNty6+6r
# naz5d/+TG3cbODQ9YbzotkHh4+efWmJr2b1DDh4S4xoaiJnPB/P0H+Mzg1Pn4MwR
# 4l5xQOpB596iCPtWRDaGtuDN3zF/316f26UXIFOhMsG9c3qPD0cUnlzWn1Z9VCTP
# Se+bDfzzFvlUDXIw7fqVvYhc+f6kfBdwYv5fuQS1XN9C3jASKep54y7kG5qWydpP
# 1Y9td28tCwuvRbQUHvvoMR/wGrIZ1qOOZrqA4EsJVUDyRjLP0orPQZEwfkpg2rQi
# IZKLfcUDc4G1jvzKZ4u4D4U2mn6zDVRVdjPW5gvW4jqhD8vTPfwXe8z1msJotnHS
# +hQMUtqyU0TcVPV4R8VuxOTfAMsRjqi6+M4OgOP65aX7zYBSGNS9klX7YmQzL/JB
# STNyG+sO0kxjeQ/7GZ6WI8vfbH8JgTa2ANYYzsC+6pRuISdS4x647Bze6ynM4CxA
# HZbQOra8jmAaeotAC8FdmDsJJGEYGmerZqJz5qx+r752tHIqoj1JJvHr3pB1
# SIG # End signature block
